<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Case study submission editor
 *
 * @package    mod_casestudy
 * @copyright  2025 SCCA
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require_once('../../config.php');

require_once(dirname(__FILE__).'/lib.php');

$id = required_param('id', PARAM_INT); // Course module ID
$submissionid = optional_param('submissionid', 0, PARAM_INT); // Submission ID for editing
$action = optional_param('action', '', PARAM_ALPHA); // Action to perform.

// Get course module and related data
$cm = get_coursemodule_from_id('casestudy', $id, 0, false, MUST_EXIST);
$course = $DB->get_record('course', array('id' => $cm->course), '*', MUST_EXIST);
$casestudy = $DB->get_record('casestudy', array('id' => $cm->instance), '*', MUST_EXIST);

require_login($course, true, $cm);

$context = context_module::instance($cm->id);

// Check if user can submit OR manage submissions (for teachers/managers).
if (!has_capability('mod/casestudy:submit', $context) && !has_capability('mod/casestudy:managesubmissions', $context)) {
    throw new required_capability_exception($context, 'mod/casestudy:submit', 'nopermissions', '');
}

$PAGE->set_url('/mod/casestudy/submission.php', array('id' => $cm->id, 'submissionid' => $submissionid));
$PAGE->set_title(format_string($casestudy->name));
$PAGE->set_heading(format_string($course->fullname));
$PAGE->set_context($context);
$PAGE->set_course($course);
$PAGE->set_cm($cm);


// Initialize managers
$fieldmanager = \mod_casestudy\local\field_manager::instance($casestudy->id);
$submissionmanager = \mod_casestudy\local\submission_manager::instance($casestudy->id, $casestudy, $cm);

// Get or create submission
if ($action) {

    switch ($action) {
        case 'delete':
            require_sesskey();
            $submission = $submissionmanager->get_submission_record($submissionid);
            if (!empty($submission) && $submissionmanager->can_delete_submission($submission, $USER->id)) {
                $submissionmanager->delete_submission($submission);
                redirect(new moodle_url('/mod/casestudy/view.php', array('id' => $cm->id)),
                        get_string('submissiondeleted', 'mod_casestudy'), null, \core\output\notification::NOTIFY_SUCCESS);
            } else {
                redirect(new moodle_url('/mod/casestudy/view.php', array('id' => $cm->id)),
                        get_string('cannotdeletesubmission', 'mod_casestudy'), null, \core\output\notification::NOTIFY_WARNING);
            }
            break;
        case 'reattempt':
            require_sesskey();
            $submission = $submissionmanager->get_submission_record($submissionid);
            if (!empty($submission) && $submissionmanager->can_reattempt_submission($submission, $USER->id)) {
                $submissionmanager->recreate_submission($submission);
                redirect(new moodle_url('/mod/casestudy/view.php', array('id' => $cm->id)),
                        get_string('submissionreattempted', 'mod_casestudy'), null, \core\output\notification::NOTIFY_SUCCESS);
            } else {
                redirect(new moodle_url('/mod/casestudy/view.php', array('id' => $cm->id)),
                        get_string('cannotreattemptsubmission', 'mod_casestudy'), null, \core\output\notification::NOTIFY_WARNING);
            }
            break;
        default:
            throw new moodle_exception('invalidaction', 'mod_casestudy');
    }
}


$submission = $submissionmanager->get_or_create_user_submission($USER->id, $submissionid);

if (!$submission) {
    redirect(new moodle_url('/mod/casestudy/view.php', array('id' => $cm->id)),
            get_string('submissionlevelreached', 'mod_casestudy'), null, \core\output\notification::NOTIFY_ERROR);
}

$editing = !empty($submissionid) && $submission && $submission->id == $submissionid;

// Get fields
$fields = $fieldmanager->get_fields();

if (empty($fields)) {
    redirect(new moodle_url('/mod/casestudy/view.php', array('id' => $cm->id)),
            get_string('nofields', 'mod_casestudy'), null, \core\output\notification::NOTIFY_ERROR);
}

// Check if user can edit this submission
if ($editing && !$submissionmanager->can_edit_submission($submission, $USER->id, true)) {
    throw new moodle_exception('cannotedisubmission', 'mod_casestudy');
}

// Get existing submission data for form
$submissiondata = [];
if ($submission && $submission->id) {
    $submissiondata = $submissionmanager->get_submission_form_data($submission->id);
}

// Create form
// Prepare form data
$formdata = array(
    'id' => $cm->id,
    'submissionid' => $submission->id
);

// Add existing submission data

$cmdata = $DB->get_record('casestudy', array('id' => $cm->instance), '*', MUST_EXIST);

$form = new \mod_casestudy\local\forms\submission_edit_form($PAGE->url, array(
    'fields' => $fields,
    'fieldmanager' => $fieldmanager,
    'editing' => $editing,
    'cmdata' => $cmdata,
));

$form->update_formdata_beforeset($submissiondata);

$formdata = array_merge($formdata, $submissiondata);

$form->set_data($formdata);

// Handle form submission
if ($form->is_cancelled()) {
    redirect(new moodle_url('/mod/casestudy/view.php', array('id' => $cm->id)));
} else if ($data = $form->get_data()) {

    $issubmit = $form->is_finish_submission($data);
    $submissionid = $editing ? $submission->id : 0;

    try {

        // Process submission through manager
        $submission = $submissionmanager->process_form_submission(
            $USER->id, $data, $submissionid, $issubmit, $form
        );

        $redirecturl = $form->is_save_and_add_another($data)
            ? new moodle_url('/mod/casestudy/submission.php', array('id' => $cm->id))
            : new moodle_url('/mod/casestudy/view.php', array('id' => $cm->id));

        $message = $issubmit
            ? get_string('submissionsubmitted', 'mod_casestudy')
            : get_string('submissionsaved', 'mod_casestudy');

        redirect($redirecturl, $message, null, \core\output\notification::NOTIFY_SUCCESS);

    } catch (Exception $e) {
        debugging('Error processing submission: ' . $e->getMessage(), DEBUG_DEVELOPER);
        $form->set_data($data);

        echo $OUTPUT->notification(get_string('submissionerror', 'mod_casestudy'), 'error');
    }
}

//pop up.
$totalunanswered = 0;
$requiresubmit = !empty($casestudy->requiresubmit) ? 1 : 0;
$PAGE->requires->js_call_amd('mod_casestudy/submission_confirmation','init',[$totalunanswered, $requiresubmit]);

echo $OUTPUT->header();

// Page heading
if ($editing) {
    echo $OUTPUT->heading(get_string('editcasestudy', 'mod_casestudy'));
} else {
    echo $OUTPUT->heading(get_string('addcasestudy', 'mod_casestudy'));
}

$form->display();

echo $OUTPUT->footer();
