<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Download case study submission as PDF
 *
 * @package    mod_casestudy
 * @copyright  2025 SCCA
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require_once(dirname(dirname(dirname(__FILE__))).'/config.php');
require_once(dirname(__FILE__).'/lib.php');
require_once($CFG->libdir.'/pdflib.php');

$id = required_param('id', PARAM_INT); // Course module ID.
$submissionid = required_param('submissionid', PARAM_INT); // Casestudy submission ID.

// Get course module and related data.
$cm = get_coursemodule_from_id('casestudy', $id, 0, false, MUST_EXIST);
$course = $DB->get_record('course', array('id' => $cm->course), '*', MUST_EXIST);
$casestudy = $DB->get_record('casestudy', array('id' => $cm->instance), '*', MUST_EXIST);

require_login($course, true, $cm);

$context = context_module::instance($cm->id);

// Get submission.
$submission = $DB->get_record('casestudy_submissions', array('id' => $submissionid));
if (!$submission || $submission->casestudyid != $casestudy->id) {
    throw new moodle_exception('invalidsubmission', 'mod_casestudy');
}

// Check permissions - only graders can download PDFs.
$cangrade = has_capability('mod/casestudy:grade', $context);
if (!$cangrade) {
    throw new moodle_exception('nopermissions', 'error');
}

// Get submission data.
$user = $DB->get_record('user', array('id' => $submission->userid));
$fields = $DB->get_records('casestudy_fields', array('casestudyid' => $casestudy->id), 'sortorder');
$content = $DB->get_records('casestudy_content', array('submissionid' => $submission->id), '', 'fieldid, content, contentformat');

// Initialize field manager.
$fieldmanager = \mod_casestudy\local\field_manager::instance($casestudy->id);

$html = mod_casestudy_output_fragment_load_submission([
    'casestudy' => $casestudy, 'submissionid' => $submission->id, 'user' => $user, 'context' => $context]);

// Convert pluginfile.php URLs to local file paths for TCPDF
$html = casestudy_convert_pluginfile_urls_for_pdf($html, $context);

// Create PDF.
$doc = new pdf();
$doc->SetTitle(get_string('casestudysubmission', 'mod_casestudy') . ': ' . fullname($user));
$doc->SetAuthor(fullname($user));
$doc->SetCreator('Moodle mod_casestudy');
$doc->SetSubject($casestudy->name);

// Set margins
$doc->SetMargins(15, 15, 15);
$doc->SetHeaderMargin(5);
$doc->SetFooterMargin(10);

// Set auto page breaks
$doc->SetAutoPageBreak(true, 25);

// Add page
$doc->AddPage();

// Write HTML content
$doc->writeHTML($html, true, false, true, false, '');

// Output PDF.
$filename = clean_filename($casestudy->name . '_' . fullname($user) . '_submission.pdf');
$doc->Output($filename, 'D');
