<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Template class for case study submissions
 *
 * @package    mod_casestudy
 * @copyright  2025 SCCA
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace mod_casestudy;

use stdClass;
use html_writer;
use moodle_url;
use mod_casestudy\local\casestudy;
use mod_casestudy\local\submission;

defined('MOODLE_INTERNAL') || die();

class template {

    private $casestudyinstance;

    protected $casestudy;

    private $cm;

    private $context;

    public function __construct(casestudy $casestudy, $cm, $context) {
        $this->casestudyinstance = $casestudy;
        $this->casestudy = $this->casestudyinstance->get_casestudy_record();
        $this->cm = $cm;
        $this->context = $context;
    }

    public function render_submission(submission $submission, $fields = null, $contents = null, $grade = null) {
        global $DB, $OUTPUT, $USER;

        $templatecontent = $this->casestudy->singletemplate;

        if (empty($templatecontent)) {
            return null; // Signal to use default rendering.
        }

        // Get the formatted data from submission's export_for_template
        $submissiondata = $submission->export_for_template($OUTPUT);

        $submissionrecord = $submission->get_submission();
        $user = $DB->get_record('user', ['id' => $submissionrecord->userid]);

        // Build replacements using the export_for_template data
        $replacements = $this->get_replacements_from_export($submissiondata, $submissionrecord, $user, $grade);
        $output = $this->parse_template($templatecontent, $replacements);

        if (!empty($this->casestudy->csstemplate)) {
            $output = html_writer::tag('style', $this->casestudy->csstemplate) . $output;
        }

        return $output;
    }

    /**
     * Update the field html for the template replacements.
     *
     * @param array $fielddata The field data from export_for_template
     * @return string The HTML for the field value
     */
    protected function update_field_html($fielddata) {
        $attributes = [
            'class' => 'field-display mt-2 field-' . $fielddata['type']
        ];

        $valuehtml = '';
        if (!empty($fielddata['hasvalue'])) {
            $valuehtml = $fielddata['value'];
        } else {
            $valuehtml = html_writer::tag(
            'em',
            get_string('novalue', 'mod_casestudy'),
            ['class' => 'text-muted']
            );
        }

        return \html_writer::div(
            \html_writer::div($valuehtml, 'field-value'), null, $attributes);
    }

    /**
     * Get replacements for template tags using export_for_template data.
     *
     * @param array $submissiondata The data from submission->export_for_template()
     * @param stdClass $submission The submission record
     * @param stdClass $user The user record
     * @param stdClass|null $grade The grade record, if any
     *
     * @return array The replacements for the template tags.
     */
    private function get_replacements_from_export($submissiondata, $submission, $user, $grade) {
        global $OUTPUT;

        $replacements = [];

        // User information from casestudyinfo.
        $casestudyinfo = $submissiondata['casestudyinfo'];
        $userpicture = $OUTPUT->user_picture($user, ['size' => 50, 'class' => 'userpicture']);
        $replacements['[[userpicture]]'] = $userpicture;
        $replacements['[[user]]'] = $casestudyinfo['student'];
        $replacements['[[userid]]'] = $user->id;

        // Submission information.
        $replacements['[[timesubmitted]]'] = $casestudyinfo['timecreated'] ?: get_string('notsubmitted', 'mod_casestudy');
        $replacements['[[timecreated]]'] = $casestudyinfo['timecreated'];
        $replacements['[[timemodified]]'] = $casestudyinfo['timemodified'];
        $replacements['[[status]]'] = $casestudyinfo['statuslabel'];
        $replacements['[[attempt]]'] = $submission->attempt;

        // Field data from export_for_template - already formatted!
        if (isset($submissiondata['fieldsbyshortname'])) {

            foreach ($submissiondata['fieldsbyshortname'] as $shortname => $fielddata) {
                // Use the already rendered value from export_for_template.
                $fieldcontent = $this->update_field_html($fielddata);// ['value'];

                // Support both field name and shortname.
                $replacements['[[' . $fielddata['name'] . ']]'] = $fieldcontent;
                $replacements['[[' . $fielddata['name'] . '#name]]'] = $fielddata['name'];
                $replacements['[[' . $fielddata['name'] . '#id]]'] = $fielddata['id'];
                $replacements['[[' . $shortname . ']]'] = $fieldcontent;

                // Also support lowercase without spaces version.
                $replacements['[[' . strtolower(str_replace(' ', '', $fielddata['name'])) . ']]'] = $fieldcontent;
                $replacements['[[' . strtolower(str_replace(' ', '', $fielddata['name'])) . '#name]]'] = $fielddata['name'];
                $replacements['[[' . strtolower(str_replace(' ', '', $fielddata['name'])) . '#id]]'] = $fielddata['id'];
            }
        }

        // Grade information.
        if ($grade) {
            $replacements['[[grade]]'] = $this->get_grade_display($grade);
            $replacements['[[feedback]]'] = format_text($grade->feedback, $grade->feedbackformat);
            $replacements['[[grader]]'] = $this->get_grader_name($grade->graderid);
            $replacements['[[gradetime]]'] = userdate($grade->timemodified);
        } else {
            $replacements['[[grade]]'] = '';
            $replacements['[[feedback]]'] = '';
            $replacements['[[grader]]'] = '';
            $replacements['[[gradetime]]'] = '';
        }

        // Action buttons.
        $replacements['##edit##'] = $this->get_edit_button($submission);
        $replacements['##delete##'] = $this->get_delete_button($submission);
        $replacements['##view##'] = $this->get_view_button($submission);

        return $replacements;
    }

    /**
     * Parse the template content and replace tags with actual values.
     *
     * @param string $template The template content
     * @param array $replacements The replacements for the template tags
     *
     * @return string The parsed template with replacements
     */
    private function parse_template($template, $replacements) {
        $output = $template;

        foreach ($replacements as $tag => $value) {
            $output = str_replace($tag, $value, $output);
        }

        $output = preg_replace('/\[\[.*?\]\]/', '', $output);
        $output = preg_replace('/##.*?##/', '', $output);

        return $output;
    }

    public function get_default_template($fields) {
        $template = '<div class="casestudy-submission-single">';
        $template .= '<div class="submission-header">';
        $template .= '<h3>[[userpicture]] [[user]]</h3>';
        $template .= '<div class="submission-meta">';
        $template .= '<span class="status">Status: [[status]]</span> | ';
        $template .= '<span class="attempt">Attempt: [[attempt]]</span> | ';
        $template .= '<span class="submitted">Submitted: [[timesubmitted]]</span>';
        $template .= '</div>';
        $template .= '</div>';

        $template .= '<div class="submission-content">';
        foreach ($fields as $field) {
            $template .= '<div class="field-wrapper field-' . $field->id . '">';
            $template .= '<b> [[' . $field->name . '#name]] </b>';
            $template .= '[[' . $field->name . ']]';
            $template .= '</div>';
        }
        $template .= '</div>';

        $template .= '<div class="submission-grade">';
        $template .= '[[grade]]';
        $template .= '[[feedback]]';
        $template .= '</div>';

        $template .= '</div>';

        return $template;
    }

    private function get_grade_display($grade) {
        global $OUTPUT;

        $gradevalue = $grade->grade;
        if ($gradevalue == 0) {
            $gradetext = get_string('unsatisfactory', 'mod_casestudy');
            $gradeclass = 'badge badge-danger';
        } else if ($gradevalue == 1) {
            $gradetext = get_string('satisfactory', 'mod_casestudy');
            $gradeclass = 'badge badge-success';
        } else {
            return '';
        }

        return html_writer::span($gradetext, $gradeclass);
    }

    private function get_grader_name($graderid) {
        global $DB;

        if ($this->casestudy->hidegrader) {
            return get_string('grader', 'mod_casestudy');
        }

        $grader = $DB->get_record('user', ['id' => $graderid]);
        return fullname($grader);
    }

    private function get_edit_button($submission) {
        global $USER;

        if ($submission->userid != $USER->id) {
            return '';
        }

        $canresubmit = in_array($submission->status, ['draft', 'awaiting_resubmission']);
        if (!$canresubmit) {
            return '';
        }

        $url = new moodle_url('/mod/casestudy/submission.php', [
            'id' => $this->cm->id,
            'submissionid' => $submission->id,
            'action' => 'edit'
        ]);

        return html_writer::link($url, get_string('edit'), ['class' => 'btn btn-secondary']);
    }

    private function get_delete_button($submission) {
        global $USER;

        // Use submission_manager to check if user can delete
        $submissionmanager = \mod_casestudy\local\submission_manager::instance(
            $this->casestudy->id,
            $this->casestudy,
            $this->cm
        );

        if (!$submissionmanager->can_delete_submission($submission, $USER->id)) {
            return '';
        }

        $url = new moodle_url('/mod/casestudy/submission.php', [
            'id' => $this->cm->id,
            'submissionid' => $submission->id,
            'action' => 'delete',
            'sesskey' => sesskey()
        ]);

        return html_writer::link($url, get_string('delete'), ['class' => 'btn btn-danger']);
    }

    private function get_view_button($submission) {
        $url = new moodle_url('/mod/casestudy/view_casestudy.php', [
            'id' => $this->cm->id,
            'submissionid' => $submission->id
        ]);

        return html_writer::link($url, get_string('view'), ['class' => 'btn btn-primary']);
    }

    public function get_available_tags($fields) {
        $tags = [
            'fields' => [],
            'fieldattr' => [],
            'user' => [
                '[[userpicture]]' => get_string('tag_userpicture', 'mod_casestudy'),
                '[[user]]' => get_string('tag_user', 'mod_casestudy'),
                '[[userid]]' => get_string('tag_userid', 'mod_casestudy'),
            ],
            'submission' => [
                '[[timesubmitted]]' => get_string('tag_timesubmitted', 'mod_casestudy'),
                '[[timecreated]]' => get_string('tag_timecreated', 'mod_casestudy'),
                '[[timemodified]]' => get_string('tag_timemodified', 'mod_casestudy'),
                '[[status]]' => get_string('tag_status', 'mod_casestudy'),
                '[[attempt]]' => get_string('tag_attempt', 'mod_casestudy'),
            ],
            'grade' => [
                '[[grade]]' => get_string('tag_grade', 'mod_casestudy'),
                '[[feedback]]' => get_string('tag_feedback', 'mod_casestudy'),
                '[[grader]]' => get_string('tag_grader', 'mod_casestudy'),
                '[[gradetime]]' => get_string('tag_gradetime', 'mod_casestudy'),
            ],
        ];


        foreach ($fields as $field) {
            $tags['fields']['[[' . $field->name . ']]'] = $field->name;
            $tags['fieldattr']['[[' . $field->name . '#title]]'] = $field->name . ' ' . get_string('tag_title', 'mod_casestudy');
            $tags['fieldattr']['[[' . $field->name . '#id]]'] = $field->name . ' ' . get_string('tag_id', 'mod_casestudy');
        }

        return $tags;
    }
}